<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\Therapists\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Http\Requests\SuperAdmin\Therapists\UpdateRequest;
use App\Imports\SuperAdmin\TherapistsImport;
use App\Http\Resources\Web\TherapistsResource;
use Inertia\Inertia;
use App\Models\Therapist;
use App\Models\TherapistCommunity;
use App\Models\TherapistMainCategory;
use App\Models\TherapistCategory;
use App\Models\PricingPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use Session;
use App\Http\Controllers\NotificationSettingsController;
use App\Models\Role;

class TherapistsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.index');
        $this->middleware('permission:therapist.add', ['only' => ['store']]);
        $this->middleware('permission:therapist.edit', ['only' => ['update']]);
        $this->middleware('permission:therapist.delete', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.export', ['only' => ['export']]);
        $this->middleware('permission:therapist.import', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null)
    {
        if ($req != null) {
            $therapists =  Therapist::withAll();
            if ($req->trash && $req->trash == 'with') {
                $therapists =  $therapists->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapists =  $therapists->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapists = $therapists->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapists = $therapists->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapists = $therapists->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapists = $therapists->OrderBy('is_approved', 'ASC');
            }
            if ($export != null) { // for export do not paginate
                $therapists = $therapists->get();
                return $therapists;
            }
            $therapists = $therapists->get();
            return $therapists;
        }
        $therapists = Therapist::withAll()->OrderBy('is_approved', 'ASC')->get();
        return $therapists;
    }


    /*********View All Therapists  ***********/
    public function index(Request $request)
    {
        $therapists = $this->getter($request);
        $therapist_communitys = TherapistCommunity::approved()->active()->get();
        return view('super_admins.therapists.index', compact('therapists', 'therapist_communitys'));
    }

    /*********View Create Form of Therapist  ***********/
    public function create()
    {
        $therapist_communitys = TherapistCommunity::active()->approved()->get();
        $therapist_categories = TherapistCategory::active()->get();
        $pricing_plans = PricingPlan::therapist()->get();
        return view('super_admins.therapists.create', compact('pricing_plans', 'therapist_communitys', 'therapist_categories'));
    }

    /*********Store Therapist  ***********/
    public function store(CreateRequest $request)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if (!$request->is_featured) {
                $data['is_featured'] = 0;
            }
            if (!$request->is_premium) {
                $data['is_premium'] = 0;
            }
            $data['image'] = uploadCroppedFile($request, 'image', 'therapists');

            $therapist = Therapist::create($data);
            $user = User::where('email', $request->email)->first();
            if ($user) {
                $user->roles()->attach(['therapist']);
                $therapist->update(['user_id' => $user->id]);
            } else {
                $user = $therapist->user()->create([
                    'name' => $therapist->name,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                ]);
                $user->markEmailAsVerified();
                $therapist->update(['user_id' => $user->id]);
                $user->roles()->attach(['therapist']);
            }
            $therapist->therapist_categories()->attach($request->therapist_category_ids);
            $therapist->save();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapists.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapists.index')->with('message', 'Therapist Created Successfully')->with('message_type', 'success');
    }

    /*********View Therapist  ***********/
    public function show(Therapist $therapist)
    {
        $therapist = Therapist::withAll()->find($therapist->id);
        return view('super_admins.therapists.show', compact('therapist'));
    }

    /*********View Edit Form of Therapist  ***********/
    public function edit(Therapist $therapist)
    {
        $pricing_plans = PricingPlan::therapist()->get();
        $therapist_categories = TherapistCategory::active()->get();
        $therapist_communitys = TherapistCommunity::active()->approved()->get();
        return view('super_admins.therapists.edit', compact('therapist', 'pricing_plans', 'therapist_communitys', 'therapist_categories'));
    }

    /*********Update Therapist  ***********/
    public function update(UpdateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if (!$request->is_featured) {
                $data['is_featured'] = 0;
            }
            if (!$request->is_premium) {
                $data['is_premium'] = 0;
            }
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapists', $therapist->image);
            } else {
                $data['image'] = $therapist->image;
            }
            if (isset($request->therapist_category_ids)) {
                $therapist->therapist_categories()->sync($request->therapist_category_ids);
            }
            $therapist->update($data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapists.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapists.index')->with('message', 'Therapist Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapists = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapists." . $extension;
        return Excel::download(new TherapistsExport($therapists), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistsImport, $file);
        return redirect()->back()->with('message', 'Blog Categories imported Successfully')->with('message_type', 'success');
    }

    public function viewBlogs(Therapist $therapist)
    {
        $therapist_blogs = $therapist->therapist_posts()->get();
        $therapist_id = $therapist->id;
        return view('super_admins.therapists.show_blogs', compact('therapist_blogs', 'therapist_id'));
    }
    public function viewEvents(Therapist $therapist)
    {
        $therapist_events = $therapist->therapist_events()->get();
        $therapist_id = $therapist->id;
        return view('super_admins.therapists.show_events', compact('therapist_events', 'therapist_id'));
    }
    public function viewSocialLinks(Therapist $therapist)
    {
        $therapist_settings = $therapist->therapist_settings()->get();
        $therapist_id = $therapist->id;
        return view('super_admins.therapists.show_social', compact('therapist_settings', 'therapist_id'));
    }

    public function profile(Request $request, $therapist)
    {
        $therapist = therapist::withChildrens()->withAll()->where('id', $therapist)->first();
        if (!$therapist) {
            abort(404);
        }
        $therapist = new TherapistsResource($therapist);
        return Inertia::render('Therapists/Profile', [
            'therapist' => $therapist
        ]);
    }


    /*********Soft DELETE Therapist ***********/
    public function destroy(Therapist $therapist)
    {
        $user = $therapist->user;
        $user->roles()->detach([Role::$Therapist]);
        if (!$user->hasRole(Role::$TherapistCommunity) && !$user->hasRole(Role::$Customer) && !$user->hasRole(Role::$SuperAdmin)) {
            $user->delete();
        }
        $therapist->delete();
        return redirect()->back()->with('message', 'Therapist Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Therapist ***********/
    public function destroyPermanently(Request $request, $therapist)
    {
        $therapist = Therapist::withTrashed()->find($therapist);
        if ($therapist) {
            if ($therapist->trashed()) {
                if ($therapist->image && file_exists(public_path($therapist->image))) {
                    unlink(public_path($therapist->image));
                }
                $therapist->forceDelete();
                return redirect()->back()->with('message', 'Therapist Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Therapist is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Therapist Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Therapist***********/
    public function restore(Request $request, $therapist)
    {
        $therapist = Therapist::withTrashed()->find($therapist);
        if ($therapist->trashed()) {
            $therapist->restore();
            return redirect()->back()->with('message', 'Therapist Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Therapist Not Found')->with('message_type', 'error');
        }
    }
    /*********Approve Therapist ***********/
    public function approve(Therapist $therapist)
    {
        if (!$therapist->is_approved) {
            $therapist->update(['is_approved' => 1, 'approved_at' => now()]);
        }

        NotificationSettingsController::fireNotificationEvent($therapist,'approve_or_reject_therapist',[[$therapist]],'super_admin/therapists','Therapist Approved Successfully');


        // NotificationSettingsController::fireNotificationEvent($therapist, 'approve_or_reject_therapist', [$therapist]);

        return redirect()->back()->with('message', 'Therapist Approved Successfully')->with('message_type', 'success');
    }
   public function bulkActionTherapists(Request $request, $type)
    {
        if ($type == 'approve') {
            Therapist::whereIn('id', $request->selected_ids)->update([
                'is_approved' => 1
            ]);
        } elseif ($type == 'disapprove') {
            Therapist::whereIn('id', $request->selected_ids)->update([
                'is_approved' => 0
            ]);
        } elseif ($type == 'inactive') {
            Therapist::whereIn('id', $request->selected_ids)->update([
                'is_active' => 0
            ]);
        } elseif ($type == 'active') {
            Therapist::whereIn('id', $request->selected_ids)->update([
                'is_active' => 1
            ]);
        } elseif ($type == 'delete') {
            foreach ($request->selected_ids as $userId) {
                $therapist = Therapist::where('id', $userId)->first();
                $this->destroy($therapist);
            }
        } elseif ($type == 'feature') {
            Therapist::whereIn('id', $request->selected_ids)->update([
                'is_featured' => 1
            ]);
        } else {
            Session::flash('message', 'Some Thing Went Wrong !');
            return response()->json('Success', 200);
        }
        Session::flash('message', 'Updated Successfully');
        return response()->json('Success', 200);
    }
}
